<?php
if (!defined('ABSPATH')) exit;

/**
 * JW3 Importer C vers00o corrigida e limpa
 * --------------------------------------
 * - Corrige termos (status, tipo, rea, budget, localiza0400o)
 * - Cria termos corretamente usando term_id
 * - Suporta taxonomias hierarchical e non-hierarchical
 * - Salva rea com fallback inteligente
 * - Salva comodidades, negocia0400o, localiza0400o, pre04o
 * - Salva imagens via URLs diretas (sem sideload pesado)
 * - Salva RAW + coords + iframe
 * - Melhora logs
 */

class JW3_Importer {

    private $api;
    private $post_type = 'spaciaz_project';
    public $batch_size = 5;
    private $pointer_opt = 'jw3_page_pointer';
    private $update_opt  = 'jw3_update_existing';

    public function __construct($api, $args = array()){
        $this->api = $api;
        if (isset($args['post_type']))   $this->post_type   = $args['post_type'];
        if (isset($args['batch_size']))  $this->batch_size  = intval($args['batch_size']);
    }

    /* ================================================================
       BATCH IMPORT
       ================================================================ */
    public function run_batch_import(){
        $page = intval(get_option($this->pointer_opt, 1));
        jw3_log("Batch start page {$page}");

        $props = $this->api->get_properties($page, $this->batch_size, true);

        if (is_wp_error($props)) {
            jw3_log("API error: ".$props->get_error_message());
            return;
        }

        if (empty($props) || !is_array($props)) {
            jw3_log("No properties found  resetting pointer to 1.");
            update_option($this->pointer_opt, 1);
            return;
        }

        foreach ($props as $p){

            $external = '';
            if (!empty($p['codigo']))        $external = (string)$p['codigo'];
            elseif (!empty($p['id_imovel'])) $external = (string)$p['id_imovel'];
            elseif (!empty($p['id']))        $external = (string)$p['id'];

            if (!$external) continue;

            try {
                $this->process_safe($external, $p);
            } catch (\Exception $e){
                jw3_log("ERROR importing {$external}: ".$e->getMessage());
            }
        }

        update_option($this->pointer_opt, $page + 1);
        jw3_log("Batch complete, next page ".($page + 1));
    }

    /* ================================================================
       REIMPORT BY POST ID
       ================================================================ */
    public function reimport_by_post_id($post_id){
        $raw = get_post_meta($post_id, '_jetimob_raw', true);
        if (!$raw) return false;

        $external = '';
        if (!empty($raw['codigo']))        $external = (string)$raw['codigo'];
        elseif (!empty($raw['id_imovel'])) $external = (string)$raw['id_imovel'];
        elseif (!empty($raw['id']))        $external = (string)$raw['id'];

        if (!$external) $external = "reimport_{$post_id}";

        jw3_log("Reimporting post {$post_id} ({$external})");

        return $this->process_safe($external, $raw, $post_id);
    }

    /* ================================================================
       TERM CREATION (CORE FIX)
       ================================================================ */
    private function ensure_term($term_name, $taxonomy){
        $term_name = trim((string)$term_name);
        if ($term_name === '') return false;

        if (!taxonomy_exists($taxonomy)) {
            jw3_log("ensure_term: taxonomy does not exist: {$taxonomy}");
            return false;
        }

        // Check existing
        $exists = term_exists($term_name, $taxonomy);

        if ($exists && !is_wp_error($exists)) {
            return is_array($exists) ? intval($exists['term_id']) : intval($exists);
        }

        // Try insert - may race, so handle error
        $insert = wp_insert_term($term_name, $taxonomy);

        if (is_wp_error($insert)) {
            // If another process created it, try to fetch again by name
            jw3_log("TERM ERROR [{$taxonomy}] '{$term_name}': ".$insert->get_error_message());
            $exists_retry = term_exists($term_name, $taxonomy);
            if ($exists_retry && !is_wp_error($exists_retry)) {
                return is_array($exists_retry) ? intval($exists_retry['term_id']) : intval($exists_retry);
            }
            return false;
        }

        jw3_log("Created term '{$term_name}' in taxonomy {$taxonomy}");

        return intval($insert['term_id']);
    }

    /* ================================================================
       SET TERMS USING TERM ID
       ================================================================ */
    private function set_terms_auto($post_id, $terms, $taxonomy){
        if (empty($terms)) return;
        if (!taxonomy_exists($taxonomy)) {
            jw3_log("Taxonomy missing: {$taxonomy}");
            return;
        }

        if (!is_array($terms)) $terms = array($terms);

        $term_ids = [];

        foreach ($terms as $t){
            $t = trim((string)$t);
            if ($t === '') continue;

            $id = $this->ensure_term($t, $taxonomy);
            if ($id) $term_ids[] = intval($id);
        }

        if (empty($term_ids)) return;

        // remove duplicates
        $term_ids = array_values(array_unique($term_ids));

        // Set terms by ID (wp_set_post_terms accepts IDs)
        $res = wp_set_post_terms($post_id, $term_ids, $taxonomy, false);

        if (is_wp_error($res)) {
            jw3_log("SET_TERM ERROR [{$taxonomy}]: ".$res->get_error_message());
        } else {
            jw3_log("SET {$taxonomy} -> ".json_encode($term_ids));
        }
    }

    /* ================================================================
       PROCESS SINGLE PROPERTY
       ================================================================ */
    private function process_safe($external, $data, $force_post_id = 0){

        /* -------------------------------------------------------------
           FIND OR CREATE POST
        ------------------------------------------------------------- */

        if ($force_post_id){
            $post_id = $force_post_id;
        } else {
            $existing = get_posts([
                'post_type'   => $this->post_type,
                'meta_key'    => 'jetimob_id',
                'meta_value'  => $external,
                'numberposts' => 1,
                'fields'      => 'ids'
            ]);

            $post_id = !empty($existing) ? $existing[0] : 0;
        }

        /* -------------------------------------------------------------
           POST CONTENT
        ------------------------------------------------------------- */

        $title =
            (!empty($data['titulo_anuncio'])) ? $data['titulo_anuncio'] :
            ((!empty($data['titulo'])) ? $data['titulo'] : "Imvel {$external}");

        $content = '';
        if (!empty($data['observacoes'])) $content .= wp_kses_post($data['observacoes']);
        if (!empty($data['descricao_anuncio'])) $content .= "\n\n".wp_kses_post($data['descricao_anuncio']);

        $args = [
            'post_title'   => wp_strip_all_tags($title),
            'post_content' => $content,
            'post_status'  => 'publish',
            'post_type'    => $this->post_type
        ];

        if ($post_id){
            if (get_option($this->update_opt, 'yes') === 'yes'){
                $args['ID'] = $post_id;
                wp_update_post($args);
            }
        } else {
            $post_id = wp_insert_post($args);
        }

        if (!$post_id || is_wp_error($post_id)) return false;

        /* -------------------------------------------------------------
           SAVE IDENTIFIERS
        ------------------------------------------------------------- */
        update_post_meta($post_id, 'jetimob_id', $external);
        update_post_meta($post_id, 'project_code', $external);

        /* -------------------------------------------------------------
           TAXONOMIAS
        ------------------------------------------------------------- */

        if (!empty($data['status'])) $this->set_terms_auto($post_id, $data['status'], 'project_status');

        if (!empty($data['subtipo'])) $this->set_terms_auto($post_id, $data['subtipo'], 'project_type');
        elseif (!empty($data['tipo'])) $this->set_terms_auto($post_id, $data['tipo'], 'project_type');

        if (!empty($data['endereco_bairro'])) $this->set_terms_auto($post_id, $data['endereco_bairro'], 'project_location');
        elseif (!empty($data['endereco_cidade'])) $this->set_terms_auto($post_id, $data['endereco_cidade'], 'project_location');

        /* -------------------------------------------------------------
           PRE05O
        ------------------------------------------------------------- */
        if (!empty($data['valor_venda']) && !empty($data['valor_venda_visivel'])){
            $price = floatval($data['valor_venda']);
            update_post_meta($post_id, '_price', $price);
            update_post_meta($post_id, 'project_price', $price);

            $price_term = "R$ ".number_format($price, 0, ',', '.');
            $this->set_terms_auto($post_id, $price_term, 'project_budget');
        }

        /* -------------------------------------------------------------
           09REA C fallback
        ------------------------------------------------------------- */

        $area = null;

        if (!empty($data['area_privativa']) && floatval($data['area_privativa']) > 0) {
            $area = $data['area_privativa'];
        } elseif (!empty($data['area_total']) && floatval($data['area_total']) > 0) {
            $area = $data['area_total'];
        } elseif (!empty($data['area_util']) && floatval($data['area_util']) > 0) {
            $area = $data['area_util'];
        } elseif (!empty($data['tipologia']) && is_array($data['tipologia'])){
            $tip = implode(" ", $data['tipologia']);
            if (preg_match('/(\d{2,4})\s?m/i', $tip, $m)) $area = $m[1];
        } elseif (!empty($data['imovel_comodidades'])){
            if (preg_match('/(\d{2,4})\s?m/i', $data['imovel_comodidades'], $m2)) $area = $m2[1];
        }

        if ($area && floatval($area) > 0){
            $area_num = floatval(str_replace([',','m','m05','m2'], '', strtolower($area)));
            update_post_meta($post_id, 'project_area_raw', $area_num);

            $label = "{$area_num} m05";

            $this->set_terms_auto($post_id, $label, 'project_area');

            jw3_log("AREA OK {$external} => {$label}");
        } else {
            jw3_log("NO AREA FOUND for {$external}");
        }

        /* -------------------------------------------------------------
           UNIDADES BADEN (tags)
        ------------------------------------------------------------- */
        if (!empty($data['tags'])){
            $tags = is_array($data['tags']) ? $data['tags'] : explode(',', $data['tags']);
            $tags = array_map('trim', $tags);
            $this->set_terms_auto($post_id, $tags, 'unidades_baden');
        }

        /* -------------------------------------------------------------
           COMODIDADES
        ------------------------------------------------------------- */
        if (!empty($data['imovel_comodidades'])) update_post_meta($post_id, 'imovel_comodidades', $data['imovel_comodidades']);
        if (!empty($data['condominio_comodidades'])) update_post_meta($post_id, 'condominio_comodidades', $data['condominio_comodidades']);
        /* ================================================================
   CONVERTER COMODIDADES  _spaciaz_project_features
   ================================================================ */

$features = [];

$raw_features = [];

// junta todas as comodidades num nico array
if (!empty($data['imovel_comodidades'])) {
    $raw_features[] = $data['imovel_comodidades'];
}
if (!empty($data['condominio_comodidades'])) {
    $raw_features[] = $data['condominio_comodidades'];
}

if (!empty($raw_features)) {

    // separar por vrgula, limpar e normalizar
    $items = explode(',', implode(',', $raw_features));
    $items = array_map('trim', $items);
    $items = array_filter($items);

    foreach ($items as $item) {
        $features[] = [
            'icon'  => 'fas fa-check-circle',
            'title' => $item,
        ];
    }

    update_post_meta($post_id, '_spaciaz_project_features', $features);

    jw3_log("FEATURES UPDATED (" . count($features) . ") -> {$external}");
}


        /* -------------------------------------------------------------
           NEGOCIA0501O
        ------------------------------------------------------------- */
        if (!empty($data['contrato'])){
            $c = strtolower($data['contrato']);
            $neg = '';
            if (strpos($c, 'compra') !== false && strpos($c, 'loca') !== false) $neg = 'compra_locacao';
            elseif (strpos($c, 'compra') !== false) $neg = 'compra';
            elseif (strpos($c, 'loca') !== false) $neg = 'locacao';

            if ($neg) update_post_meta($post_id, '_negociacao_tipo', $neg);
        }

        /* -------------------------------------------------------------
           IMAGENS / PLANTAS (URLs diretas)
        ------------------------------------------------------------- */
        $photos = [];
        if (!empty($data['imagens'])){
            foreach ((array)$data['imagens'] as $img){
                if (!empty($img['link'])) $photos[] = esc_url_raw($img['link']);
            }
        }

        $plans = [];
        if (!empty($data['plantas'])){
            foreach ((array)$data['plantas'] as $pl){
                if (!empty($pl['link'])) $plans[] = esc_url_raw($pl['link']);
            }
        }

        update_post_meta($post_id, 'jetimob_images', $photos);
        update_post_meta($post_id, 'jetimob_plans', $plans);
        update_post_meta($post_id, 'jetimob_thumb', !empty($photos[0]) ? $photos[0] : '');

        /* -------------------------------------------------------------
           VIDEO
        ------------------------------------------------------------- */
        if (!empty($data['videos'][0]['href'])) update_post_meta($post_id, 'spaciaz_video_url', esc_url_raw($data['videos'][0]['href']));

        /* -------------------------------------------------------------
           MAPA
        ------------------------------------------------------------- */
        if (!empty($data['latitude']) && !empty($data['longitude'])){
            update_post_meta($post_id, 'latitude',  $data['latitude']);
            update_post_meta($post_id, 'longitude', $data['longitude']);

            $iframe = '<iframe src="https://www.google.com/maps?q='.esc_attr($data['latitude']).','.esc_attr($data['longitude']).'&output=embed" '.
                      'width="100%" height="400" style="border:0;" allowfullscreen loading="lazy"></iframe>';

            update_post_meta($post_id, 'spaciaz_location_map_iframe', $iframe);
        }

        /* -------------------------------------------------------------
           RAW SAVE
        ------------------------------------------------------------- */
        update_post_meta($post_id, '_jetimob_raw', $data);

        jw3_log("Processed {$external} (post {$post_id})");

        return $post_id;
    }
}
