<?php
if (!defined('ABSPATH')) exit;

/**
 * Jetimob API Wrapper – versão final
 * ------------------------------------
 * - Comunicação estável com timeout seguro
 * - Erros detalhados no log
 * - Retorno padronizado sempre em array
 * - Failover caso Jetimob envie estrutura inesperada
 */

class Jetimob_API {

    private $base_url = 'https://api.jetimob.com/webservice';
    private $api_key;
    private $timeout = 30;

    public function __construct($api_key){
        $this->api_key = trim($api_key);
    }

    /**
     * GET – Buscar imóveis
     */
    public function get_properties($page = 1, $page_size = 5, $only_active = true){

        if (empty($this->api_key)) {
            return new WP_Error('jetimob_no_key', 'Jetimob key not set');
        }

        $endpoint = sprintf(
            '%s/%s/imoveis/todos',
            $this->base_url,
            rawurlencode($this->api_key)
        );

        $params = array(
            'page'     => intval($page),
            'pageSize' => intval($page_size),
        );

        if ($only_active) {
            $params['v'] = 'ativo';
        }

        $url = $endpoint . '?' . http_build_query($params);

        $response = wp_remote_get($url, array(
            'timeout' => $this->timeout,
            'headers' => array(
                'Accept'      => 'application/json',
                'User-Agent'  => 'JW3-Importer/3.0'
            )
        ));

        /* CONNECTION ERROR */
        if (is_wp_error($response)) {
            jw3_log("API ERROR: " . $response->get_error_message());
            return $response;
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);

        /* HTTP ERROR */
        if ($code !== 200) {
            jw3_log("API HTTP {$code}: " . substr($body, 0, 300));
            return new WP_Error('jetimob_api_error', "HTTP {$code}");
        }

        /* PARSE JSON */
        $data = json_decode($body, true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            jw3_log("JSON ERROR: " . json_last_error_msg());
            return new WP_Error('jetimob_json_error', json_last_error_msg());
        }

        /* NORMAL RETURN */
        if (isset($data['imoveis']) && is_array($data['imoveis'])) {
            return $data['imoveis'];
        }

        /* SOME IMPORTS RETURN AS PURE ARRAY */
        if (is_array($data) && isset($data[0])) {
            return $data;
        }

        /* FAILOVER: UNKNOWN FORMAT */
        jw3_log("API WARNING: Formato inesperado -> " . substr(json_encode($data), 0, 300));

        return is_array($data) ? $data : array();
    }
}
