<?php

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

use Elementor\Controls_Manager;
use Elementor\Repeater;
use Elementor\Group_Control_Image_Size;

class Spaciaz_Video_Tabs extends Elementor\Widget_Base
{

    public function get_name()
    {
        return 'spaciaz-video-tabs';
    }

    public function get_title()
    {
        return esc_html__('Spaciaz Video Tabs', 'spaciaz');
    }

    public function get_icon()
    {
        return 'eicon-youtube';
    }

    public function get_script_depends()
    {
        return ['spaciaz-elementor-video-tabs',];
    }

    protected function register_controls()
    {
        $this->start_controls_section(
            'section_videos',
            [
                'label' => esc_html__('General', 'spaciaz'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $repeater = new Repeater();

        $repeater->add_control(
            'title',
            [
                'label' => esc_html__('Video Title', 'spaciaz'),
                'type' => Controls_Manager::TEXT,
            ]
        );
        $repeater->add_control(
            'video_link',
            [
                'label' => esc_html__('Video Link', 'spaciaz'),
                'type' => Controls_Manager::TEXT,
                'description' => esc_html__('Support video from Youtube, Vimeo, or a direct .mp4 file.', 'spaciaz'),
                'placeholder' => esc_html__('https://your-link.com', 'spaciaz'),
            ]
        );

        $repeater->add_control(
            'background_video',
            [
                'label' => esc_html__('Video Thumbnail', 'spaciaz'),
                'type' => Controls_Manager::MEDIA,
            ]
        );

        $this->add_control(
            'videos',
            [
                'label' => esc_html__('Videos', 'spaciaz'),
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'title_field' => '{{{ title }}}',
            ]
        );

        $this->add_control(
            'video_style',
            [
                'label' => esc_html__('Video Style', 'spaciaz'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'container',
                'options' => [
                    'natural' => esc_html__('Aspect Ratio (Natural)', 'spaciaz'),
                    'container' => esc_html__('Full Container', 'spaciaz'),
                ],
                'render_type' => 'template',
                'prefix_class' => 'elementor-video-style-'
            ]
        );

        $this->add_control(
            'relative_id',
            [
                'label' => esc_html__('Relative Container', 'spaciaz'),
                'type' => Controls_Manager::TEXT,
                'description' => esc_html__('Add the ID or class selector of the relative element (e.g., #my-section or .my-section).', 'spaciaz'),
                'condition' => [
                    'video_style' => 'container',
                ]
            ]
        );

        $this->add_responsive_control(
            'video_height',
            [
                'label' => esc_html__('Video Height', 'spaciaz'),
                'type' => Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'min' => 1,
                        'max' => 1000,
                    ],
                ],
                'size_units' => ['px', 'custom'],
                'selectors' => [
                    '{{WRAPPER}} .video-container' => 'height: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'video_style' => 'natural',
                ]
            ]
        );

        $this->add_group_control(
            Group_Control_Image_Size::get_type(),
            [
                'name' => 'background_video',
                'default' => 'full',
                'separator' => 'none',
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'section_navigation_style',
            [
                'label' => esc_html__('Tabs Nav', 'spaciaz'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_responsive_control(
            'image_width',
            [
                'label' => esc_html__('Item Width', 'spaciaz'),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px', 'custom'],
                'selectors' => [
                    '{{WRAPPER}} .elementor-video-tabs-image img' => 'width: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'image_height',
            [
                'label' => esc_html__('Item Height', 'spaciaz'),
                'type' => Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'min' => 1,
                        'max' => 1000,
                    ],
                ],
                'size_units' => ['px', 'custom'],
                'selectors' => [
                    '{{WRAPPER}} .elementor-video-tabs-image img' => 'height: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'image_border_radius',
            [
                'label' => esc_html__('Border Radius', 'spaciaz'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .elementor-video-tabs-image img' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}',
                ],
            ]
        );

        $this->add_responsive_control(
            'item_nav_spacing',
            [
                'label' => esc_html__('Spacing', 'spaciaz'),
                'type' => Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'default' => [
                    'size' => 20
                ],
                'size_units' => ['px', 'em'],
                'selectors' => [
                    '{{WRAPPER}} .video-tabs-nav' => 'gap:{{SIZE}}{{UNIT}}',
                ],
            ]
        );

        $this->add_control(
            'nav_vertical',
            [
                'label' => esc_html__('Nav Vertical', 'spaciaz'),
                'type' => Controls_Manager::CHOOSE,
                'label_block' => false,
                'default' => 'bottom',
                'options' => [
                    'top' => [
                        'title' => esc_html__('Top', 'spaciaz'),
                        'icon' => 'eicon-v-align-top',
                    ],
                    'bottom' => [
                        'title' => esc_html__('Bottom', 'spaciaz'),
                        'icon' => 'eicon-v-align-bottom',
                    ],
                ],
                'condition' => [
                    'video_style' => 'natural',
                ]
            ]
        );

        $this->add_responsive_control(
            'nav_vertical_value',
            [
                'type' => Controls_Manager::SLIDER,
                'show_label' => false,
                'size_units' => ['px', '%'],
                'range' => [
                    'px' => [
                        'min' => -1000,
                        'max' => 1000,
                        'step' => 1,
                    ],
                    '%' => [
                        'min' => -100,
                        'max' => 100,
                    ],
                ],
                'default' => [
                    'unit' => '%',
                    'size' => 50,
                ],
                'selectors' => [
                    '{{WRAPPER}} .video-tabs-nav' => 'top: unset; bottom: unset; {{nav_vertical.value}}: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'video_style' => 'natural',
                ]
            ]
        );

        $this->add_control(
            'nav_horizontal',
            [
                'label' => esc_html__('Nav Horizontal', 'spaciaz'),
                'type' => Controls_Manager::CHOOSE,
                'label_block' => false,
                'default' => 'left',
                'options' => [
                    'left' => [
                        'title' => esc_html__('Left', 'spaciaz'),
                        'icon' => 'eicon-h-align-left',
                    ],
                    'right' => [
                        'title' => esc_html__('Right', 'spaciaz'),
                        'icon' => 'eicon-h-align-right',
                    ],
                    'condition' => [
                        'video_style' => 'natural',
                    ]
                ],
            ]
        );
        $this->add_responsive_control(
            'nav_horizontal_value',
            [
                'type' => Controls_Manager::SLIDER,
                'show_label' => false,
                'size_units' => ['px', 'em', '%', 'custom'],
                'range' => [
                    'px' => [
                        'min' => -1000,
                        'max' => 1000,
                        'step' => 1,
                    ],
                    '%' => [
                        'min' => -100,
                        'max' => 100,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 80,
                ],
                'selectors' => [
                    '{{WRAPPER}} .video-tabs-nav' => 'left: unset; right: unset;{{nav_horizontal.value}}: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'video_style' => 'natural',
                ]
            ]
        );

        $this->end_controls_section();

    }

    protected function render()
    {
        $settings = $this->get_settings_for_display();
        $this->add_render_attribute('wrapper', 'class', 'video-tabs-element');
        if (!empty($settings['relative_id'])) {
            $this->add_render_attribute('wrapper', 'data-relative', $settings['relative_id']);
        }
        if (!empty($settings['videos']) && is_array($settings['videos'])) {
            $control_html = '';
            ?>
            <div <?php $this->print_render_attribute_string('wrapper'); ?>>
                <div class="video-tabs-content">
                    <?php
                    foreach ($settings['videos'] as $key => $video) {
                        $active_class = $key === 0 ? 'active' : '';
                        if (!empty($video['video_link'])) {
                            ?>
                            <div class="video-tab-content <?php echo esc_attr($active_class); ?>"
                                 id="video-tab-<?php echo esc_attr($key); ?>">
                                <?php
                                $this->convertVideoUrlToEmbed($video['video_link'], 1);
                                ?>
                            </div>
                            <?php
                            ob_start();
                            ?>
                            <div class="video-tab-nav-item <?php echo esc_attr($active_class); ?>"
                                 data-tab="video-tab-<?php echo esc_attr($key); ?>">
                                <?php $this->render_image($settings, $video); ?>
                            </div>
                            <?php
                            $control_html .= ob_get_clean();
                        }
                    }
                    ?>
                </div>
                <div class="video-tabs-nav">
                    <?php printf('%s', $control_html); ?>
                </div>
            </div>
            <?php
        }
    }

    private function render_image($settings, $video)
    {
        if (!empty($video['background_video']['url'])) :
            ?>
            <div class="elementor-video-tabs-image">
                <?php
                $video['background_video_size'] = $settings['background_video_size'];
                $video['background_video_custom_dimension'] = $settings['background_video_custom_dimension'];
                echo Group_Control_Image_Size::get_attachment_image_html($video, 'background_video');
                ?>
            </div>
        <?php
        endif;
    }

    private function convertVideoUrlToEmbed($url, $autoplay = 0)
    {
        $videourl = '';
        $class = '';
        if ($url && !empty($url)) {
            if (preg_match('/https?:\/\/(www\.)?youtube\.com\/watch\?v=([^&]+)/', $url, $matches)) {
                $videoId = $matches[2];
                $videourl = "https://www.youtube.com/embed/{$videoId}?loop=1&playlist={$videoId}&mute=1&modestbranding=1&rel=0&controls=0&enablejsapi=1&disablekb=1&fs=0&autoplay={$autoplay}&preload=none";
                $class = 'iframe-youtube';
            } elseif (preg_match('/https?:\/\/(www\.)?vimeo\.com\/(\d+)/', $url, $matches)) {
                $videoId = $matches[2];
                $videourl = "https://player.vimeo.com/video/{$videoId}?background=1&loop=1&muted=1&controls=0&autoplay={$autoplay}";
            } elseif (preg_match('/\.mp4$/', $url)) {
                $videourl = $url;
            } else {
                $videourl = $url;
            }

            if (!empty($videourl)) {
                if (preg_match('/\.mp4$/', $videourl)) {
                    $autoplay_html = $autoplay ? 'autoplay' : '';
                    echo '<div class="video-container">
                        <video width="100%" height="auto" loop ' . $autoplay_html . ' playsinline muted crossorigin="anonymous" preload="auto">
                            <source src="' . $videourl . '" type="video/mp4">
                        </video>
                      </div>';
                } else {
                    echo '<div class="video-container"><iframe class="' . esc_attr($class) . '" src="' . $videourl . '" loading="lazy" allow="autoplay; fullscreen; picture-in-picture; clipboard-write; encrypted-media" referrerpolicy="strict-origin-when-cross-origin"></iframe></div>';
                }
            }
        }
    }

}

$widgets_manager->register(new Spaciaz_Video_Tabs());
