<?php

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

use Elementor\Controls_Manager;
use Elementor\Group_Control_Typography;

/**
 * Class Spaciaz_Elementor_Service
 */
class Spaciaz_Elementor_Service extends Spaciaz_Base_Widgets_Swiper {

    public function get_name() {
        return 'spaciaz-services';
    }

    public function get_title() {
        return esc_html__('Spaciaz Service', 'spaciaz');
    }

    /**
     * Get widget icon.
     *
     * Retrieve testimonial widget icon.
     *
     * @return string Widget icon.
     * @since  1.0.0
     * @access public
     *
     */
    public function get_icon() {
        return 'eicon-posts-grid';
    }

    public function get_categories() {
        return array('spaciaz-addons');
    }

    public function get_script_depends() {
        return ['spaciaz-elementor-services', 'spaciaz-elementor-swiper'];
    }

    protected function register_controls() {
        $this->start_controls_section(
            'section_query',
            [
                'label' => esc_html__('Service', 'spaciaz'),
                'tab'   => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'posts_per_page',
            [
                'label'   => esc_html__('Posts Per Page', 'spaciaz'),
                'type'    => Controls_Manager::NUMBER,
                'default' => 6,
            ]
        );

        $this->add_control(
            'service_style',
            [
                'label'        => esc_html__('Style', 'spaciaz'),
                'type'         => \Elementor\Controls_Manager::SELECT,
                'options'      => [
                    'service-style-1' => esc_html__('Style 1', 'spaciaz'),
                    'service-style-2' => esc_html__('Style 2', 'spaciaz'),
                    'service-style-3' => esc_html__('Style 3', 'spaciaz'),
                    'service-style-4' => esc_html__('Style 4', 'spaciaz'),
                    'service-style-5' => esc_html__('Style 5', 'spaciaz'),
                    'service-style-6' => esc_html__('Style 6', 'spaciaz'),
                ],
                'render_type'  => 'template',
                'default'      => 'service-style-1',
                'prefix_class' => 'elementor-'
            ]
        );

        $this->add_control('raw_style_6',[
            'type' => Controls_Manager::RAW_HTML,
            'raw' => esc_html__('To use this feature, drag the Spaciaz Service Content Toggle widget to the page and set the Query the same as the Service widget. The content will then automatically sync and change with each item.','spaciaz'),
            'content_classes' => 'elementor-descriptor',
            'condition' => [
                'service_style' => 'service-style-6',
            ],
        ]);

        $this->add_control(
            'enable_sticky',
            [
                'label' => esc_html__('Enable Sticky', 'spaciaz'),
                'type'  => Controls_Manager::SWITCHER,
                'prefix_class' => 'elementor-service-enable-sticky-',

                'conditions'   => [
                    'relation' => 'and',
                    'terms'    => [
                        [
                            'name'     => 'service_style',
                            'operator' => '==',
                            'value'    => 'service-style-3',
                        ],
                        [
                            'name'     => 'enable_carousel',
                            'operator' => '!==',
                            'value'    => 'yes',
                        ],
                    ],
                ]
            ]
        );

        $this->add_responsive_control(
            'effect_offset',
            [
                'label'     => esc_html__('Effect Offset', 'spaciaz'),
                'type'      => Controls_Manager::SLIDER,
                'range'     => [
                    'px' => [
                        'min' => 0,
                        'max' => 500,
                    ],
                ],

                'selectors' => [
                    '{{WRAPPER}}.elementor-service-enable-sticky-yes .grid-item' => 'top: {{SIZE}}{{UNIT}};',
                ],
                'condition'  => ['enable_sticky' => 'yes']
            ]
        );

        $this->add_control(
            'includes_ids',
            [
                'label'       => esc_html__('Includes', 'spaciaz'),
                'type'         => 'spaciaz_query',
                'autocomplete' => [
                    'object' => 'spaciaz_services',
                ],
                'label_block' => true,
                'multiple'    => true,
            ]
        );

        $this->add_control(
            'excludes_ids',

            [
                'label'       => esc_html__('Excludes', 'spaciaz'),
                'type'         => 'spaciaz_query',
                'autocomplete' => [
                    'object' => 'spaciaz_services',
                ],
                'label_block' => true,
                'multiple'    => true,
            ]
        );

        $this->add_control(
            'orderby',
            [
                'label'   => esc_html__('Order By', 'spaciaz'),
                'type'    => Controls_Manager::SELECT,
                'default' => 'post_date',
                'options' => [
                    'post_date'  => esc_html__('Date', 'spaciaz'),
                    'post_title' => esc_html__('Title', 'spaciaz'),
                    'menu_order' => esc_html__('Menu Order', 'spaciaz'),
                    'rand'       => esc_html__('Random', 'spaciaz'),
                ],
            ]
        );

        $this->add_control(
            'order',
            [
                'label'   => esc_html__('Order', 'spaciaz'),
                'type'    => Controls_Manager::SELECT,
                'default' => 'desc',
                'options' => [
                    'asc'  => esc_html__('ASC', 'spaciaz'),
                    'desc' => esc_html__('DESC', 'spaciaz'),
                ],
            ]
        );

        $this->add_responsive_control(
            'column',
            [
                'label'     => esc_html__('Columns', 'spaciaz'),
                'type'      => \Elementor\Controls_Manager::SELECT,
                'default'   => 3,
                'options'   => [1 => 1, 2 => 2, 3 => 3, 4 => 4, 5 => 5],
                'selectors' => [
                    '{{WRAPPER}} .d-grid' => 'grid-template-columns: repeat({{VALUE}}, 1fr)',
                ],
                'conditions'   => [
                    'relation' => 'and',
                    'terms'    => [
                        [
                            'name'     => 'service_style',
                            'operator' => '!==',
                            'value'    => 'service-style-3',
                        ],
                        [
                            'name'     => 'service_style',
                            'operator' => '!==',
                            'value'    => 'service-style-4',
                        ],
                        [
                            'name'     => 'service_style',
                            'operator' => '!==',
                            'value'    => 'service-style-6',
                        ],
                        [
                            'name'     => 'enable_carousel',
                            'operator' => '!==',
                            'value'    => 'yes',
                        ],
                    ],
                ]
            ]
        );

        $this->add_responsive_control(
            'item_spacing',
            [
                'label'      => esc_html__('Spacing', 'spaciaz'),
                'type'       => Controls_Manager::SLIDER,
                'range'      => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'default'    => [
                    'size' => 30
                ],
                'size_units' => ['px', 'em'],
                'selectors'  => [
                    '{{WRAPPER}} .d-grid' => 'grid-gap:{{SIZE}}{{UNIT}}',
                ],
                'condition'  => ['enable_carousel!' => 'yes','service_style!' => 'service-style-4']
            ]
        );

        $this->add_control(
            'enable_carousel',
            [
                'label' => esc_html__('Enable Carousel', 'spaciaz'),
                'type'  => Controls_Manager::SWITCHER,
                'conditions'   => [
                    'relation' => 'and',
                    'terms'    => [
                        [
                            'name'     => 'service_style',
                            'operator' => '!==',
                            'value'    => 'service-style-4',
                        ],
                    ],
                ]
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'section_pagination',
            [
                'label'     => esc_html__('Pagination', 'spaciaz'),
                'conditions'   => [
                    'relation' => 'and',
                    'terms'    => [
                        [
                            'name'     => 'service_style',
                            'operator' => '!==',
                            'value'    => 'service-style-4',
                        ],
                        [
                            'name'     => 'enable_carousel',
                            'operator' => '!==',
                            'value'    => 'yes',
                        ],
                    ],
                ]
            ]

        );

        $this->add_control(
            'pagination_type',
            [
                'label'   => esc_html__('Pagination', 'spaciaz'),
                'type'    => Controls_Manager::SELECT,
                'default' => '',
                'options' => [
                    ''                      => esc_html__('None', 'spaciaz'),
                    'numbers'               => esc_html__('Yes', 'spaciaz'),
                ],
            ]
        );

        $this->add_control(
            'pagination_page_limit',
            [
                'label'     => esc_html__('Page Limit', 'spaciaz'),
                'default'   => '5',
                'condition' => [
                    'pagination_type!' => '',
                ],
            ]
        );

        $this->add_control(
            'pagination_align',
            [
                'label'     => esc_html__('Alignment', 'spaciaz'),
                'type'      => Controls_Manager::CHOOSE,
                'options'   => [
                    'flex-start' => [
                        'title' => esc_html__('Left', 'spaciaz'),
                        'icon'  => 'eicon-text-align-left',
                    ],
                    'center'     => [
                        'title' => esc_html__('Center', 'spaciaz'),
                        'icon'  => 'eicon-text-align-center',
                    ],
                    'flex-end'   => [
                        'title' => esc_html__('Right', 'spaciaz'),
                        'icon'  => 'eicon-text-align-right',
                    ],
                ],
                'default'   => 'center',
                'selectors' => [
                    '{{WRAPPER}} ul.page-numbers' => 'justify-content: {{VALUE}};',
                ],
                'condition' => [
                    'pagination_type!' => '',
                ],
            ]
        );

        $this->end_controls_section();

        // Title
        $this->start_controls_section(
            'service_title_style',
            [
                'label' => esc_html__('Service Style', 'spaciaz'),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'style_content',
            [
                'type'  => Controls_Manager::HEADING,
                'label' => esc_html__('Content', 'spaciaz'),

            ]
        );

        $this->add_responsive_control(
            'content_padding',
            [
                'label'      => esc_html__('Padding', 'spaciaz'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .service-info' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    '{{WRAPPER}} .service-style-2 .service-content' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    '{{WRAPPER}} .service-style-3 .service-content' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'content_margin',
            [
                'label'      => esc_html__('Margin', 'spaciaz'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .service-inner' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'style_title',
            [
                'type'  => Controls_Manager::HEADING,
                'label' => esc_html__('Title', 'spaciaz'),

            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name'     => 'title_typography',
                'selector' => '{{WRAPPER}} .service-title',
            ]
        );

        $this->add_control(
            'title_color',
            [
                'label' => esc_html__('Color', 'spaciaz'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .service-title a' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'style_icon',
            [
                'type'  => Controls_Manager::HEADING,
                'label' => esc_html__('Icon', 'spaciaz'),
                'condition'  => ['service_style' => 'service-style-2']
            ]
        );

        $this->add_responsive_control(
            'icon_size',
            [
                'label'      => esc_html__('Size', 'spaciaz'),
                'type'       => Controls_Manager::SLIDER,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .icon-wrap img' => 'width: {{SIZE}}{{UNIT}};',
                ],
                'condition'  => ['service_style' => 'service-style-2']
            ]
        );

        $this->add_responsive_control(
            'icon_spacing',
            [
                'label'      => esc_html__('Margin', 'spaciaz'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .icon-wrap' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'condition'  => ['service_style' => 'service-style-2']
            ]
        );

        $this->add_responsive_control(
            'height_image',
            [
                'label'      => esc_html__('Image Height', 'spaciaz'),
                'type'       => Controls_Manager::SLIDER,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .service-post-thumbnail' => 'height: {{SIZE}}{{UNIT}}; padding-top: 0;',
                ],
            ]
        );

        $this->end_controls_section();

        $this->add_control_carousel(['enable_carousel' => 'yes','service_style!' => 'service-style-4']);

    }

    protected function render() {
        $settings = $this->get_settings_for_display();
        $this->add_render_attribute('wrapper', 'class', 'elementor-service-wrapper');

        $this->get_data_elementor_columns();

        $this->add_render_attribute('item', 'class', 'service-item ' . esc_attr($settings['service_style']));

        $allowed_styles = [
            'service-style-1',
            'service-style-2',
            'service-style-3',
            'service-style-4',
            'service-style-5',
            'service-style-6',
        ];

        $style = $settings['service_style'] ?? 'service-style-1';

        if ( ! in_array( $style, $allowed_styles, true ) ) {
            $style = 'service-style-1';
        }

        $query = $this->query_posts();

        if (!$query->found_posts) {
            return;
        }
        $count = 1;

        ?>
        <div <?php $this->print_render_attribute_string('wrapper'); ?>>
            <div <?php $this->print_render_attribute_string('row'); ?>>
                <?php
                while ($query->have_posts()) {
                    $query->the_post();
                    set_query_var('count', ($count < 10) ? '0' . $count : $count);
                    ?>
                     <?php if ($settings['service_style'] !== 'service-style-4') { ?>
                    <div <?php $this->print_render_attribute_string('item'); ?>>
                    <?php } ?>
                        <?php get_template_part('template-parts/services/item', $style); ?>
                    <?php if ($settings['service_style'] !== 'service-style-4') { ?>
                    </div>
                    <?php } ?>
                    <?php
                    $count++;
                }
                ?>
            </div>

        </div>
        <?php $this->render_swiper_pagination_navigation();
        if ($settings['pagination_type'] && !empty($settings['pagination_type'])) {
            $this->render_loop_footer();
        }
        wp_reset_postdata();

    }

    public static function get_query_args($settings) {
        $query_args = [
            'posts_per_page'      => $settings['posts_per_page'],
            'post_type'           => 'spaciaz_services',
            'orderby'             => $settings['orderby'],
            'order'               => $settings['order'],
            'ignore_sticky_posts' => 1,
            'post__in'            => $settings['includes_ids'],
            'post__not_in'        => $settings['excludes_ids'],
            'post_status'         => 'publish',
        ];

        if (is_front_page()) {
            $query_args['paged'] = (get_query_var('page')) ? get_query_var('page') : 1;
        } else {
            $query_args['paged'] = (get_query_var('paged')) ? get_query_var('paged') : 1;
        }

        return $query_args;
    }

    public function query_posts() {
        $query_args = $this->get_query_args($this->get_settings());
        return new WP_Query($query_args);
    }

    protected function render_loop_footer() {

        $parent_settings = $this->get_settings();
        if ('' === $parent_settings['pagination_type']) {
            return;
        }

        $page_limit = $this->query_posts()->max_num_pages;
        if ('' !== $parent_settings['pagination_page_limit']) {
            $page_limit = min($parent_settings['pagination_page_limit'], $page_limit);
        }

        if (2 > $page_limit) {
            return;
        }

        $this->add_render_attribute('pagination', 'class', 'elementor-pagination');

        $has_numbers = in_array($parent_settings['pagination_type'], ['numbers']);

        $links = [];

        if ($has_numbers) {
            $links = paginate_links([
                'type'               => 'list',
                'current'            => $this->get_current_page(),
                'total'              => $page_limit,
                'prev_text'          => '<i class="spaciaz-icon-chevron-left"></i>',
                'next_text'          => '<i class="spaciaz-icon-chevron-right"></i>',
                'before_page_number' => '<span class="elementor-screen-only">' . esc_html__('Page', 'spaciaz') . '</span>',
            ]);
        }

        ?>
        <nav class="pagination">
            <div class="nav-links">
                <?php printf('%s', $links); ?>
            </div>
        </nav>
        <?php
    }

    public function get_current_page() {
        if ('' === $this->get_settings('pagination_type')) {
            return 1;
        }

        return max(1, get_query_var('paged'), get_query_var('page'));
    }

}

$widgets_manager->register(new Spaciaz_Elementor_Service());