<?php

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

use Elementor\Controls_Manager;
use Elementor\Group_Control_Typography;

/**
 * Class Spaciaz_Elementor_Service
 */
class Spaciaz_Elementor_Service_Content_Toggle extends \Elementor\Widget_Base
{

    public function get_name()
    {
        return 'spaciaz-services-content-toggle';
    }

    public function get_title()
    {
        return esc_html__('Spaciaz Service Content Toggle', 'spaciaz');
    }

    /**
     * Get widget icon.
     *
     * Retrieve testimonial widget icon.
     *
     * @return string Widget icon.
     * @since  1.0.0
     * @access public
     *
     */
    public function get_icon()
    {
        return 'eicon-select';
    }

    public function get_categories()
    {
        return array('spaciaz-addons');
    }

    protected function register_controls()
    {
        $this->start_controls_section(
            'section_query',
            [
                'label' => esc_html__('Service', 'spaciaz'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'posts_per_page',
            [
                'label' => esc_html__('Posts Per Page', 'spaciaz'),
                'type' => Controls_Manager::NUMBER,
                'default' => 6,
            ]
        );

        $this->add_control(
            'includes_ids',
            [
                'label' => esc_html__('Includes', 'spaciaz'),
                'type' => 'spaciaz_query',
                'autocomplete' => [
                    'object' => 'spaciaz_services',
                ],
                'label_block' => true,
                'multiple' => true,
            ]
        );

        $this->add_control(
            'excludes_ids',

            [
                'label' => esc_html__('Excludes', 'spaciaz'),
                'type' => 'spaciaz_query',
                'autocomplete' => [
                    'object' => 'spaciaz_services',
                ],
                'label_block' => true,
                'multiple' => true,
            ]
        );

        $this->add_control(
            'orderby',
            [
                'label' => esc_html__('Order By', 'spaciaz'),
                'type' => Controls_Manager::SELECT,
                'default' => 'post_date',
                'options' => [
                    'post_date' => esc_html__('Date', 'spaciaz'),
                    'post_title' => esc_html__('Title', 'spaciaz'),
                    'menu_order' => esc_html__('Menu Order', 'spaciaz'),
                    'rand' => esc_html__('Random', 'spaciaz'),
                ],
            ]
        );

        $this->add_control(
            'order',
            [
                'label' => esc_html__('Order', 'spaciaz'),
                'type' => Controls_Manager::SELECT,
                'default' => 'desc',
                'options' => [
                    'asc' => esc_html__('ASC', 'spaciaz'),
                    'desc' => esc_html__('DESC', 'spaciaz'),
                ],
            ]
        );

        $this->end_controls_section();

        // Content style
        $this->start_controls_section(
            'section_style_description',
            [
                'label' => esc_html__('Description', 'spaciaz'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'description_typography',
                'selector' => '{{WRAPPER}} .description',
            ]
        );

        $this->add_control(
            'description_color',
            [
                'label' => esc_html__('Color', 'spaciaz'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .description' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'description_spacing',
            [
                'size_units' => ['px', 'em', '%'],
                'label' => esc_html__('Spacing', 'spaciaz'),
                'type' => Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 300,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .description' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'section_style_image',
            [
                'label' => esc_html__('Image', 'spaciaz'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'image_border_radius',
            [
                'label' => esc_html__('Border Radius', 'spaciaz'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .service-post-thumbnail' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'image_margin',
            [
                'label' => esc_html__('Margin', 'spaciaz'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .service-post-thumbnail' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'height_image',
            [
                'label'      => esc_html__('Image Height', 'spaciaz'),
                'type'       => Controls_Manager::SLIDER,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .service-post-thumbnail' => 'height: {{SIZE}}{{UNIT}}; padding-top: 0;',
                ],
            ]
        );

        $this->end_controls_section();

    }

    protected function render()
    {
        $settings = $this->get_settings_for_display();
        $this->add_render_attribute('wrapper', 'class', 'elementor-service-content-toggle-wrapper');

        $query = $this->query_posts();

        if (!$query->found_posts) {
            return;
        }
        $first = 1;
        ?>
        <div <?php $this->print_render_attribute_string('wrapper'); ?>>
            <?php
            while ($query->have_posts()) {
                $query->the_post();
                $class = $first == 1 ? 'active' : '';
                ?>
                <div class="service-content-style-6 service-content-<?php echo esc_attr(get_the_ID() . ' '.$class); ?>">
                    <div class="description">
                        <?php echo get_the_excerpt(); ?>
                    </div>
                    <div class="service-post-thumbnail">
                        <?php if (has_post_thumbnail()) : ?>
                            <?php the_post_thumbnail('large'); ?>
                        <?php endif; ?>
                    </div><!-- .post-thumbnail -->
                </div>
                <?php
                $first = 0;
            }
            ?>

        </div>
        <?php
        wp_reset_postdata();

    }

    public static function get_query_args($settings)
    {
        $query_args = [
            'posts_per_page' => $settings['posts_per_page'],
            'post_type' => 'spaciaz_services',
            'orderby' => $settings['orderby'],
            'order' => $settings['order'],
            'ignore_sticky_posts' => 1,
            'post__in' => $settings['includes_ids'],
            'post__not_in' => $settings['excludes_ids'],
            'post_status' => 'publish',
        ];

        return $query_args;
    }

    public function query_posts()
    {
        $query_args = $this->get_query_args($this->get_settings());
        return new WP_Query($query_args);
    }

}

$widgets_manager->register(new Spaciaz_Elementor_Service_Content_Toggle());