<?php

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

use Elementor\Controls_Manager;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Repeater;
use Elementor\Group_Control_Image_Size;

class Spaciaz_Elementor_Iconbox_Tab extends Spaciaz_Base_Widgets_Swiper
{
    /**
     * Get widget name.
     *
     * Retrieve iconbox widget name.
     *
     * @return string Widget name.
     * @since  1.0.0
     * @access public
     *
     */
    public function get_name()
    {
        return 'spaciaz-iconboxs-tab';
    }

    /**
     * Get widget title.
     *
     * Retrieve iconbox widget title.
     *
     * @return string Widget title.
     * @since  1.0.0
     * @access public
     *
     */
    public function get_title()
    {
        return esc_html__('Spaciaz Iconbox Tab', 'spaciaz');
    }

    /**
     * Get widget icon.
     *
     * Retrieve iconbox widget icon.
     *
     * @return string Widget icon.
     * @since  1.0.0
     * @access public
     *
     */
    public function get_icon()
    {
        return 'eicon-tabs';
    }

    public function get_script_depends()
    {
        return ['spaciaz-elementor-iconbox-tab'];
    }

    public function get_categories()
    {
        return array('spaciaz-addons');
    }

    /**
     * Register iconbox widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since  1.0.0
     * @access protected
     */

    protected function register_controls()
    {
        $this->start_controls_section(
            'section_iconbox',
            [
                'label' => esc_html__('Iconbox Tab', 'spaciaz'),
            ]
        );

        $repeater = new Repeater();

        $repeater->add_control(
            'tab_hover_icon',
            [
                'label' => esc_html__('Icon', 'spaciaz'),
                'type'  => Controls_Manager::ICONS,
                'skin'             => 'inline',
                'label_block'      => false,
            ]
        );

        $repeater->add_control(
            'iconbox_title',
            [
                'label' => esc_html__('Title', 'spaciaz'),
                'default' => 'Heading title',
                'type' => Controls_Manager::TEXT,
            ]
        );

        $repeater->add_control(
            'iconbox_content',
            [
                'label' => esc_html__('Content', 'spaciaz'),
                'type' => Controls_Manager::WYSIWYG,
                'default' => 'Click edit button to change this text. Lorem ipsum dolor sit amet, consectetur adipiscing elit. Ut elit tellus, luctus nec ullamcorper mattis, pulvinar dapibus leo.',
                'label_block' => true,
                'rows' => '10',
            ]
        );

        $repeater->add_control(
            'iconbox_link',
            [
                'label' => esc_html__('Link to', 'spaciaz'),
                'placeholder' => esc_html__('https://your-link.com', 'spaciaz'),
                'type' => Controls_Manager::URL,
            ]
        );
        $repeater->add_control(
            'iconbox_image',
            [
                'label' => esc_html__('Choose Image', 'spaciaz'),
                'type' => Controls_Manager::MEDIA,
                'show_label' => false,
            ]
        );

        $this->add_control(
            'iconboxs',
            [
                'label' => esc_html__('Items', 'spaciaz'),
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'title_field' => '{{{ iconbox_title }}}',
            ]
        );

        $this->add_group_control(
            Elementor\Group_Control_Image_Size::get_type(),
            [
                'name' => 'iconbox_image',
                'default' => 'full',
                'separator' => 'none',
            ]
        );


        $this->add_control(
            'view',
            [
                'label' => esc_html__('View', 'spaciaz'),
                'type' => Controls_Manager::HIDDEN,
                'default' => 'traditional',
            ]
        );
        $this->end_controls_section();


        // WRAPPER STYLE
        $this->start_controls_section(
            'section_style_iconbox_wrapper',
            [
                'label' => esc_html__('Wrapper', 'spaciaz'),
                'tab' => Controls_Manager::TAB_STYLE,

            ]
        );

        $this->add_responsive_control(
            'padding_iconbox_wrapper',
            [
                'label' => esc_html__('Padding', 'spaciaz'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .elementor-iconbox-tab-item-wrapper' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'margin_iconbox_wrapper',
            [
                'label' => esc_html__('Margin', 'spaciaz'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em'],
                'selectors' => [
                    '{{WRAPPER}} .elementor-iconbox-tab-item-wrapper' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'color_iconbox_wrapper',
            [
                'label' => esc_html__('Background Color', 'spaciaz'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .elementor-iconbox-tab-item-wrapper' => 'background: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name' => 'wrapper_border',
                'placeholder' => '1px',
                'default' => '1px',
                'selector' => '{{WRAPPER}} .elementor-iconbox-tab-item-wrapper',
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'wrapper_radius',
            [
                'label' => esc_html__('Border Radius', 'spaciaz'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .elementor-iconbox-tab-item-wrapper' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'wrapper_box_shadow',
                'selector' => '{{WRAPPER}} .elementor-iconbox-tab-item-wrapper',
            ]
        );
        $this->end_controls_section();


        // Tab STYLE
        $this->start_controls_section(
            'section_style_iconbox_tab',
            [
                'label' => esc_html__('Tab', 'spaciaz'),
                'tab' => Controls_Manager::TAB_STYLE,

            ]
        );

        $this->add_responsive_control(
            'iconbox_tab_width',
            [
                'label' => esc_html__('Width', 'spaciaz'),
                'type' => Controls_Manager::SLIDER,
                'default' => [
                    'unit' => '%',
                ],
                'tablet_default' => [
                    'unit' => '%',
                ],
                'mobile_default' => [
                    'unit' => '%',
                ],
                'size_units' => ['%', 'px', 'vw'],
                'range' => [
                    '%' => [
                        'min' => 1,
                        'max' => 100,
                    ],
                    'px' => [
                        'min' => 1,
                        'max' => 1000,
                    ],
                    'vw' => [
                        'min' => 1,
                        'max' => 100,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .iconbox-details-wrap' => 'flex-basis: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'iconbox_tab_spacing', [
            'label'      => esc_html__('Distance from content', 'spaciaz'),
            'type'       => Controls_Manager::SLIDER,
            'range'      => [
                'px' => [
                    'min' => 0,
                    'max' => 400,
                ],
            ],
            'size_units' => ['px'],
            'selectors'  => [
                '{{WRAPPER}} .elementor-iconbox-tab-item-wrapper' => 'gap: {{SIZE}}{{UNIT}}',
            ],
        ]);

        $this->add_responsive_control('iconbox_tab_space_between', [
            'label'      => esc_html__('Gap between title', 'spaciaz'),
            'type'       => Controls_Manager::SLIDER,
            'range'      => [
                'px' => [
                    'min' => 0,
                    'max' => 400,
                ],
            ],
            'size_units' => ['px'],
            'selectors'  => [
                '{{WRAPPER}} .details-tab-info' => 'padding: {{SIZE}}{{UNIT}} 0',
            ],
        ]);

        $this->add_control(
            'tab_border_color',
            [
                'label' => esc_html__( 'Border Color', 'spaciaz' ),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .details-tab-info' => 'border-color: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_section();

        // Image style
        $this->start_controls_section(
            'section_style_iconbox_image',
            [
                'label' => esc_html__('Image', 'spaciaz'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );


        $this->add_responsive_control(
            'image_width',
            [
                'label' => esc_html__('Width', 'spaciaz'),
                'type' => Controls_Manager::SLIDER,
                'default' => [
                    'unit' => '%',
                ],
                'tablet_default' => [
                    'unit' => '%',
                ],
                'mobile_default' => [
                    'unit' => '%',
                ],
                'size_units' => ['%', 'px', 'vw'],
                'range' => [
                    '%' => [
                        'min' => 1,
                        'max' => 100,
                    ],
                    'px' => [
                        'min' => 1,
                        'max' => 330,
                    ],
                    'vw' => [
                        'min' => 1,
                        'max' => 100,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .iconbox-tab-left' => 'flex-basis: {{SIZE}}{{UNIT}};',
                ],
            ]
        );
        $this->add_responsive_control(
            'image_height',
            [
                'label' => esc_html__('Height', 'spaciaz'),
                'type' => Controls_Manager::SLIDER,
                'default' => [
                    'unit' => 'px',
                ],
                'tablet_default' => [
                    'unit' => 'px',
                ],
                'mobile_default' => [
                    'unit' => 'px',
                ],
                'size_units' => ['px', 'vh'],
                'range' => [
                    'px' => [
                        'min' => 1,
                        'max' => 450,
                    ],
                    'vh' => [
                        'min' => 1,
                        'max' => 100,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .elementor-iconbox-tab-item-wrapper .elementor-iconbox-image' => 'padding-top: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'image_border_radius',
            [
                'label' => esc_html__('Border Radius', 'spaciaz'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .elementor-iconbox-tab-item-wrapper .elementor-iconbox-image' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );


        $this->end_controls_section();

        // Title.
        $this->start_controls_section(
            'section_style_title',
            [
                'label' => esc_html__('Title', 'spaciaz'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'title_color',
            [
                'label' => esc_html__('Color', 'spaciaz'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .title' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'title_typography',
                'selector' => '{{WRAPPER}} .title',
            ]
        );

        $this->add_responsive_control(
            'title_margin',
            [
                'label' => esc_html__('Margin', 'spaciaz'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

       
        // Content style
        $this->start_controls_section(
            'section_style_iconbox_content',
            [
                'label' => esc_html__('Content', 'spaciaz'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'content_bgcolor',
            [
                'label' => esc_html__('Background Color', 'spaciaz'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .elementor-iconbox-tab-item-wrapper .iconbox-content' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'content_color',
            [
                'label' => esc_html__('Color', 'spaciaz'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .content' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'content_color_hover',
            [
                'label' => esc_html__('Color Hover', 'spaciaz'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .elementor-iconbox-item-wrapper:hover .content' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'content_typography',
                'selector' => '{{WRAPPER}} .content',
            ]
        );

        $this->add_responsive_control(
            'content_margin',
            [
                'label' => esc_html__('Margin', 'spaciaz'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .iconbox-tab-right' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'content_padding',
            [
                'label' => esc_html__('Padding', 'spaciaz'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .iconbox-tab-right' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();


    }

    /**
     * Render iconbox widget output on the frontend.
     *
     * Written in PHP and used to generate the final HTML.
     *
     * @since  1.0.0
     * @access protected
     */
    protected function render()
    {
        $settings = $this->get_settings_for_display();


        if (!empty($settings['iconboxs']) && is_array($settings['iconboxs'])) {
            $this->add_render_attribute('wrapper', 'class', 'elementor-iconbox-tab-item-wrapper');
            $this->add_render_attribute('item', 'class', 'iconbox-tab-item');
            $detail_tab = '';
            ?>
            <div <?php $this->print_render_attribute_string('wrapper'); // WPCS: XSS ok. ?>>
                <div class="iconbox-tab-content">
                    <?php foreach ($settings['iconboxs'] as $iconbox): ?>
                        <div <?php $this->print_render_attribute_string('item'); // WPCS: XSS ok. ?>>
                            <div class="iconbox-content elementor-repeater-item-<?php echo esc_attr($iconbox['_id']); ?>">

                                <div class="iconbox-tab-right">
                                    <?php $this->render_image($settings, $iconbox); ?>
                                </div>

                                <?php
                                ob_start();
                                ?>

                                <div class="details-tab-info">
                                    <?php if (!empty($iconbox['tab_hover_icon']['value'])) : ?>
                                        <div class="icon">
                                            <?php \Elementor\Icons_Manager::render_icon($iconbox['tab_hover_icon'], ['aria-hidden' => 'true']); ?>
                                        </div>
                                    <?php endif; ?>
                                    <?php $iconbox_title_html = $iconbox['iconbox_title'];

                                    if (!empty($iconbox['iconbox_link']['url'])) {
                                        $iconbox_title_html = '<a href="' . esc_url($iconbox['iconbox_link']['url']) . '">' . esc_html($iconbox_title_html) . '</a>';
                                    }

                                    printf('<h5 class="title">%s</h5>', $iconbox_title_html);
                                    ?>
                                    <?php if (!empty($iconbox['iconbox_content'])) { ?>
                                        <div class="content"><?php echo sprintf('%s', $iconbox['iconbox_content']); ?></div>
                                    <?php } ?>
                                </div>
                                <?php
                                $detail_tab .= ob_get_clean();
                                ?>

                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>

                <div class="iconbox-details-wrap">
                    <?php printf('%s', $detail_tab); ?>
                </div>

            </div>
            <?php
        }
    }

    private function render_image($settings, $iconbox)
    {
        if (!empty($iconbox['iconbox_image']['url'])) :
            ?>
            <div class="elementor-iconbox-image">
                <div class="decor-border"> </div>
                <?php
                $iconbox['iconbox_image_size'] = $settings['iconbox_image_size'];
                $iconbox['iconbox_image_custom_dimension'] = $settings['iconbox_image_custom_dimension'];
                echo Group_Control_Image_Size::get_attachment_image_html($iconbox, 'iconbox_image');
                ?>
            </div>
        <?php
        endif;
    }
}

$widgets_manager->register(new Spaciaz_Elementor_Iconbox_Tab());

