<?php

namespace Elementor;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

/**
 * Elementor tabs widget.
 *
 * Elementor widget that displays vertical or horizontal tabs with different
 * pieces of content.
 *
 * @since 1.0.0
 */
class Spaciaz_Elementor_Horizontal_Scroll extends Widget_Base
{

    public function get_categories()
    {
        return array('spaciaz-addons');
    }

    /**
     * Get widget name.
     *
     * Retrieve tabs widget name.
     *
     * @return string Widget name.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_name()
    {
        return 'spaciaz-horizontal-scroll';
    }

    /**
     * Get widget title.
     *
     * Retrieve tabs widget title.
     *
     * @return string Widget title.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_title()
    {
        return esc_html__('Spaciaz Horizontal Scroll', 'spaciaz');
    }

    /**
     * Get widget icon.
     *
     * Retrieve tabs widget icon.
     *
     * @return string Widget icon.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_icon()
    {
        return 'eicon-tabs';
    }

    /**
     * Get widget keywords.
     *
     * Retrieve the list of keywords the widget belongs to.
     *
     * @return array Widget keywords.
     * @since 2.1.0
     * @access public
     *
     */
    public function get_keywords()
    {
        return ['horizontal', 'scroll'];
    }

    public function get_script_depends()
    {
        return ['spaciaz-elementor-horizontal-scroll'];
    }

    /**
     * Register tabs widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function register_controls()
    {

        $templates = Plugin::instance()->templates_manager->get_source('local')->get_items();

        $options = [
            '0' => '— ' . esc_html__('Select', 'spaciaz') . ' —',
        ];

        foreach ($templates as $template) {
            $options[$template['template_id']] = $template['title'] . ' (' . $template['type'] . ')';
        }

        $this->start_controls_section(
            'section_item',
            [
                'label' => esc_html__('Item', 'spaciaz'),
            ]
        );

        $repeater = new Repeater();

        $repeater->add_control(
            'title',
            [
                'label' => esc_html__('Title', 'spaciaz'),
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
            ]
        );

        $repeater->add_control(
            'template',
            [
                'label' => esc_html__('Choose Template', 'spaciaz'),
                'default' => 0,
                'type' => Controls_Manager::SELECT,
                'options' => $options,
                'label_block' => true,
            ]
        );

        $this->add_control(
            'items',
            [
                'label' => '',
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'title_field' => '{{{ title }}}',
            ]
        );


        $this->add_responsive_control(
            'column',
            [
                'label' => esc_html__('Columns', 'spaciaz'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 3,
                'options' => [1 => 1, 2 => 2, 3 => 3, 4 => 4, 5 => 5, 6 => 6],
                'selectors' => [
                    '{{WRAPPER}} .horizontal-scroll-inner .horizontal-scroll-item' => 'width: calc(100% / {{VALUE}});',
                ],
            ]
        );

        $this->add_responsive_control(
            'bottom_offset',
            [
                'label' => esc_html__('Offset Bottom', 'spaciaz'),
                'type' => Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 600,
                    ],
                ],
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .horizontal-scroll-inner' => 'top:calc(100% - {{SIZE}}{{UNIT}})',
                ],
            ]
        );

        $this->add_responsive_control(
            'scroll_height',
            [
                'label' => esc_html__('Scroll Height', 'spaciaz'),
                'type' => Controls_Manager::SLIDER,
                'default' => [
                    'unit' => 'vh',
                    'size' => 300,
                ],
                'size_units' => ['px', 'vh'],
                'selectors' => [
                    '{{WRAPPER}} .horizontal-scroll-wrapper' => 'height:{{SIZE}}{{UNIT}}',
                ],
            ]
        );

        $this->add_control(
            'enable_line',
            [
                'label' => esc_html__('Enable Line', 'spaciaz'),
                'type' => Controls_Manager::SWITCHER,
            ]
        );

        $this->add_control(
            'line_color',
            [
                'label'     => esc_html__('Line Color', 'spaciaz'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}} .horizontal-scroll-line .line-base' => 'background: {{VALUE}}; ',
                ],
                'condition'  => ['enable_line' => 'yes'],
            ]
        );


        $this->add_control(
            'line_color_active',
            [
                'label'     => esc_html__('Line Color Active', 'spaciaz'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}} .horizontal-scroll-line .line-active' => 'background: {{VALUE}}; ',
                ],
                'condition'  => ['enable_line' => 'yes'],
            ]
        );


        $this->add_responsive_control(
            'line_position_v',
            [
                'label' => esc_html__('Line Position Vertical', 'spaciaz'),
                'type' => Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 600,
                    ],
                ],
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .horizontal-scroll-line' => 'top:{{SIZE}}{{UNIT}}',
                ],
                'condition' => [
                    'enable_line' => 'yes'
                ],
            ]
        );

        $this->add_responsive_control(
            'line_position_h',
            [
                'label' => esc_html__('Line Position Horizontal', 'spaciaz'),
                'type' => Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 600,
                    ],
                ],
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .horizontal-scroll-line' => 'left:{{SIZE}}{{UNIT}}',
                ],
                'condition' => [
                    'enable_line' => 'yes'
                ],
            ]
        );

        $this->add_control(
            'relative_id',
            [
                'label'     => esc_html__('Relative Seciton', 'spaciaz'),
                'type'      => Controls_Manager::TEXT,
            ]
        );


        $this->end_controls_section();
    }

    /**
     * Render tabs widget output on the frontend.
     *
     * Written in PHP and used to generate the final HTML.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function render()
    {
        $settings = $this->get_settings_for_display();
        $this->add_render_attribute('row', 'class', 'd-flex horizontal-scroll-inner');
        if(!empty($settings['relative_id'])){
            $this->add_render_attribute('row', 'data-relative', $settings['relative_id']);
        }
        ?>
        <div class="horizontal-scroll-wrapper">
            <div <?php $this->print_render_attribute_string('row');?>>
                <?php
                if ($settings['enable_line'] == 'yes') {
                    ?>
                    <div class="horizontal-scroll-line">
                        <div class="line-base"></div>
                        <div class="line-active"></div>
                    </div>
                    <?php
                }
                foreach ($settings['items'] as $item) {
                    ?>
                    <div class="horizontal-scroll-item">
                        <?php
                        if (!empty($item['template'])) {
                            echo Plugin::instance()->frontend->get_builder_content_for_display($item['template']);
                        } else {
                            echo '<div class="horizontal-scroll-empty">' . esc_html__('No template selected.', 'spaciaz') . '</div>';
                        }
                        ?>
                    </div>
                    <?php
                } ?>
            </div>
        </div>

        <?php
    }
}

$widgets_manager->register(new Spaciaz_Elementor_Horizontal_Scroll());
