<?php
/**
 * Plugin Name: JW Jetimob Importer v3 (HostGator Safe Mode)
 * Description: Importador Jetimob otimizado para HostGator. Batch via cron, resolve URLs temporrias de imagens e salva dados sem sideload pesado.
 * Version: 3.0.2
 * Author: ChatGPT for Wesley
 */

if ( ! defined( 'ABSPATH' ) ) exit;

/** Diretrios */
define( 'JW3_DIR', plugin_dir_path( __FILE__ ) );
define( 'JW3_UPL_DIR', wp_upload_dir()['basedir'] . '/jw-importer-logs' );

/** Includes */
require_once JW3_DIR . 'class-api.php';
require_once JW3_DIR . 'class-importer.php';
require_once JW3_DIR . 'admin.php';

/* ============================================================
   ACTIVATION / DEACTIVATION
   ============================================================ */

register_activation_hook( __FILE__, 'jw3_activate' );
register_deactivation_hook( __FILE__, 'jw3_deactivate' );

function jw3_activate() {
    // agenda cron
    if ( ! wp_next_scheduled( 'jw3_cron_event' ) ) {
        wp_schedule_event( time(), 'jw3_every_3_minutes', 'jw3_cron_event' );
    }

    // cria diretrio de logs
    if ( ! file_exists( JW3_UPL_DIR ) ) {
        wp_mkdir_p( JW3_UPL_DIR );
    }

    // ponteiro inicial
    add_option( 'jw3_page_pointer', 1 );
}

function jw3_deactivate() {
    wp_clear_scheduled_hook( 'jw3_cron_event' );
}

/* ============================================================
   CUSTOM CRON INTERVAL
   ============================================================ */

add_filter( 'cron_schedules', function( $schedules ) {
    if ( ! isset( $schedules['jw3_every_3_minutes'] ) ) {
        $schedules['jw3_every_3_minutes'] = array(
            'interval' => 180,
            'display'  => 'Every 3 Minutes'
        );
    }
    return $schedules;
});

/* ============================================================
   FACTORY IMPORTER
   ============================================================ */

function jw3_get_importer() {
    $api_key = get_option( 'jw3_webservice_key', '' );
    $batch   = intval( get_option( 'jw3_batch_size', 5 ) );

    return new JW3_Importer(
        new Jetimob_API( $api_key ),
        array(
            'post_type'  => 'spaciaz_project',
            'batch_size' => $batch
        )
    );
}

/* ============================================================
   HELPER: run or defer importer safely (ensures taxonomies exist)
   ============================================================ */

function jw3_run_importer_safely( $importer ) {
    // list of taxonomies we depend on - if any is registered we can run immediately
    $required_taxonomies = array( 'project_area', 'project_status', 'project_type', 'project_location', 'project_budget' );

    $loaded = false;
    foreach ( $required_taxonomies as $tax ) {
        if ( taxonomy_exists( $tax ) ) {
            $loaded = true;
            break;
        }
    }

    // If taxonomies already exist, run immediately
    if ( $loaded ) {
        try {
            $importer->run_batch_import();
        } catch ( Exception $e ) {
            jw3_log( 'Importer immediate run error: ' . $e->getMessage() );
        }
        return;
    }

    // Otherwise defer to init with high priority to let theme register taxonomies
    add_action( 'init', function() use ( $importer ) {
        try {
            $importer->run_batch_import();
        } catch ( Exception $e ) {
            jw3_log( 'Importer deferred run error (init): ' . $e->getMessage() );
        }
    }, 100 );
}

/* ============================================================
   CRON RUNNER (invocado pelo evento wp-cron)
   ============================================================ */

add_action( 'jw3_cron_event', 'jw3_cron_runner' );

function jw3_cron_runner() {

    /** lock anti-execu0400o duplicada */
    $lock = get_option( 'jw3_lock', 0 );

    if ( $lock && ( time() - $lock ) < 240 ) {
        jw3_log( 'LOCKED: processo em execu0400o. Ignorando...' );
        return;
    }

    update_option( 'jw3_lock', time() );

    /** par09metros */
    $api_key = get_option( 'jw3_webservice_key', '' );
    $batch   = intval( get_option( 'jw3_batch_size', 5 ) );

    $importer = new JW3_Importer(
        new Jetimob_API( $api_key ),
        array(
            'post_type'   => 'spaciaz_project',
            'batch_size'  => $batch
        )
    );

    // run safely (immediate if tax exists, otherwise defer to init:100)
    jw3_run_importer_safely( $importer );

    update_option( 'jw3_lock', 0 );
}

/* ============================================================
   EXECUTAR 1 CICLO MANUALMENTE (bot00o admin)
   ============================================================ */

add_action( 'admin_post_jw3_run_once', function() {

    if ( ! current_user_can( 'manage_options' ) )
        wp_die( 'Unauthorized' );

    $importer = jw3_get_importer();

    // In admin screens the theme/taxonomies are usually loaded - still use safe runner
    jw3_run_importer_safely( $importer );

    wp_redirect( wp_get_referer() ?: admin_url() );
    exit;
});

/* ============================================================
   REPROCESSAR IMAGENS (N01O USA SIDLEAD)
   ============================================================ */

add_action( 'admin_post_jw3_reprocess_images', function() {

    if ( ! current_user_can( 'manage_options' ) )
        wp_die( 'Unauthorized' );

    $importer = jw3_get_importer();

    // reprocess_images is expected to run in admin where taxonomies already loaded
    if ( method_exists( $importer, 'reprocess_missing_images' ) ) {
        try {
            $importer->reprocess_missing_images( 20 );
        } catch ( Exception $e ) {
            jw3_log( 'reprocess_missing_images error: ' . $e->getMessage() );
        }
    }

    wp_redirect( wp_get_referer() ?: admin_url() );
    exit;
});

/* ============================================================
   LOGGER
   ============================================================ */

function jw3_log( $msg ) {

    if ( ! file_exists( JW3_UPL_DIR ) ) {
        wp_mkdir_p( JW3_UPL_DIR );
    }

    $file = JW3_UPL_DIR . '/import.log';
    $line = '[' . date( 'Y-m-d H:i:s' ) . '] ' . $msg . PHP_EOL;

    file_put_contents( $file, $line, FILE_APPEND | LOCK_EX );
}
